/* App State */
class AppState {
	constructor() {
		this.version = null;
		try {
			const manifest = chrome.runtime.getManifest();
			this.version = manifest?.version;
		} catch (e) {
			console.error(e);
		}
		this.state = { // persisted state
			settings: {}, // Extension specific settings, User settings are managed in web app.
			thread_sort_mode: 'date_desc', // default to newest sort mode
		};

		// settings
		this.settingsDefault = {
			server_url: "https://solfray.com", // fallback server url
			server_api_key: '',
			font_size: '0.7em',
		};
		this.settingsSchema = {
			server_url: 'string',
			server_api_key: 'string',
			font_size: 'string',
		};
		this.settingsDescriptions = {
			server_url: 'The URL of the server to send chats to.',
			server_api_key: 'Your API key for interfacing with the server.',
			font_size: 'General font size of extension text. Some elements may be larger or smaller.',
		};
		for (let key in this.settingsDefault) this.settingsSchema[key] = typeof this.settingsDefault[key];
		this.settingsSchema.server_url = 'string';

		// other session state variables (not persisted)
		this.publicKey = null; // fetched from server using api key
		this.lockedThreadId = null; // set to id if locked
		this.currentURL = null; // The current URL being viewed by the user.
		this.loadState();

		// Uses class CUN2 from thread.js
		this.threadApp = null; // set dynamically when a thread is loaded
	}

	getCurrentURL() {
		return this.currentURL || null;
	}

	setCurrentURL(url) {
		this.currentURL = null;
		$('#thread-container').empty();
		if (typeof url == 'string' && url.length > 0 && url.toLowerCase().startsWith('https://')) {
			this.currentURL = url;
			$('#thread-container').append(`<div class="col-md-8 offset-md-2"><span class="text-muted small" title="${this.currentURL}">${this.getShortURL()}</span></div>`)
		}
		return this.currentURL;
	}

	feed(arg, err = false) {
		if (err){
			console.trace(arg);	// for debugging
			$('#feed_error').removeClass('d-none');
		}else{
			$('#feed_error').addClass('d-none');
		}
		$('#feed').empty().append((arg.toString() || "&nbsp;"));
	}

	lockThread() {
		// Will prevent re-render on user navigation in browser.
		this.lockedThreadId = (this.threadApp && typeof this.threadApp == 'object')? (this.threadApp?.thread_id || null): null;
		return this.lockedThreadId ? true : false;
	}

	unlockThread() {
		// Will allow re-render on user navigation in browser.
		this.lockedThreadId = null;
		return false;
	}

	toggleThreadLock() {
		if (this.lockedThreadId) return this.unlockThread();
		return this.lockThread();
	}

	getShortURL() {
		const url_len = 40;
		const url = this.currentURL || null;
		if (!url || typeof url !== 'string') return null;
		var shortUrl = url.substring(0, url_len);
		return url.length > url_len ? shortUrl + "..." : url + "";
	}

	primeThread(chatId, replyToChatId = null, exitChatId = null) {
		return;
	}

	emptyContainers() {
		// clean slate
		$('#thread-container').empty();
		$('#reply-container').empty();
		$('#compose_container').empty();
	}

	unloadThread() {
		// destroy existing threadApp if exists
		if (this.threadApp && typeof this.threadApp == 'object') {
			this.threadApp.destroy();
			this.threadApp = null;
		}
		$('#thread-container').removeClass('thread');
		// render short url display
		this.setCurrentURL(this.getCurrentURL()); 
	}

	loadThread(chatId, authorPK = null) {
		$("#thread_id").val(chatId);
		this.emptyContainers();
		this.unloadThread();
		$('#thread-container').addClass('thread'); // required by render_threads().
		this.threadApp = new CUN2(chatId, authorPK, this.publicKey, this.getSetting('server_url'));
		this.threadApp.start((e) => { // form submit handler
			e.preventDefault();
			// serialize form
			const formDataArray = $(e.target).serializeArray();
			const dict = {};
			formDataArray.forEach(item => {
				dict[item.name] = item.value;
			});
			const lamports = parseInt(dict['lamports'] || '0') || 0;
			if(lamports > 0){ // TODO: Post to web app (DApp) to handle SOL sending
				alert("Sending SOL not via extension not implement yet.");
				return;
			}else{ // Users can send messages without tip via extension
				dict.top_chat_id = $("#thread_id").val();
				const serverURL = this.getSetting('server_url') || null;
				if (!serverURL || typeof serverURL != 'string' || serverURL.length < 1) {
					alert("Set Server URL in extension settings.");
					return;
				}
				const apiKey = this.getSetting('server_api_key') || null;
				if (!apiKey || typeof apiKey != 'string' || apiKey.length < 1) {
					alert("Set Server API Key in extension settings.");
					return;
				}
				const slash = serverURL.endsWith('/') ? '' : '/';
				const endpoint = `${serverURL}${slash}api/chats/create`;
                $.ajax({
                    url: endpoint,
                    method: "POST",
					headers: {
						'Content-Type': 'application/json',
						'Authorization': `Bearer ${apiKey}`
					},
                    data: JSON.stringify(dict),
                    contentType: "application/json",
                    success: (data) => {
                        if (data.ok) {
                            this.threadApp.build_composer_form();
                            this.threadApp.load_replies();
                        } else {
                            alert("Failed to post reply: " + data.error?.message || "Unknown error");
                        }
                    },
                    error: (xhr, status, error) => {
                        console.error("Failed to post reply:", xhr, status, error);
                        this.threadApp.end_session();
                    }
                });
			}
		});
		this.threadApp.build_composer_form();
	}

	getThreads(url = null) {
		if (this.lockedThreadId) return;
		this.emptyContainers();
		this.unloadThread();
		if (url) this.setCurrentURL(url); // will ignore non-https urls
		url = this.getCurrentURL();
		const apiKey = this.getSetting('server_api_key') || null;
		const serverURL = this.getSetting('server_url') || null;
		if(!serverURL || typeof serverURL != 'string' || serverURL.length < 1){
			this.feed("Set Server URL in extension settings.", true);
			return;
		}
		if(!apiKey || typeof apiKey != 'string' || apiKey.length < 1){
			this.feed("Set Server API Key in extension settings.", true);
			return;
		}
		// Try to fetch threads using GET request to server + Authorization header
		try{
			const slash = serverURL.endsWith('/') ? '' : '/';
			const endpoint = `${serverURL}${slash}api/threads_at_url?url=${encodeURIComponent(url)}`;
			console.log('Fetching threads from endpoint:', endpoint);
			fetch(endpoint, {
				method: 'GET',
				headers: {
					'Content-Type': 'application/json',
					'Authorization': `Bearer ${apiKey}`
				}
			}).then(response => {
				if(!response.ok) {
					response.text().then(text => {
						console.error('Fetch response not ok text:', text);
					});
					throw new Error(`Server responded with status ${response.status}`);
				}
				return response.json();
			}).then(data => {
				$('#reply-container').empty();
				if(!data || typeof data != 'object' || !Array.isArray(data.threads)){
					this.feed("Invalid response from server.", true);
					return;
				}
				if(data.threads.length < 1){
					this.feed("No threads, be the first to start one at this URL!");
					return;
				}
				const threadCount = data.threads.length;
				this.feed(`${threadCount} thread${threadCount != 1 ? 's' : ''} at this URL.`);
				for(let i=0; i<data.threads.length; i++){
					const thread = data.threads[i];
					$('#reply-container').append(`
						<div class="thread-container thread" 
							data-chat-id="${thread.chat_id}"
							data-author-alias="${(thread?.author_alias || '')}" 
							data-author-pic="${(thread?.author_pic || '')}"
							data-author-public-key="${(thread?.author_public_key || '')}"
							data-author-display-name="${(thread?.author_display_name || '')}" 
							data-content="${(thread?.content || '')}"
							data-url="${(thread?.url || '')}" 
							data-channel="${(thread?.channel_name || '')}"
							data-datetime-created="${(thread?.datetime_created || '').replace('+00:00', 'Z')}"
							data-reply-count="${(thread?.reply_count || 0)}">
						</div>
					`);
				}
				setTimeout(()=>{
					render_threads(serverURL);
				},50);
			}).catch(error => {
				console.error('Error fetching threads:', error);
				this.feed("Error fetching threads from server.", true);
			});
		}catch(e){
			this.feed("Error preparing to fetch threads from server.", true);
			$('#thread-container').empty();
			return;
		}
	}

	authenticate(){
		// Attempt to fetch my public key from the server (associated with API key)
		try{
			const serverURL = this.getSetting('server_url') || null;
			const apiKey = this.getSetting('server_api_key') || null;
			if (!serverURL || typeof serverURL != 'string' || serverURL.length < 1) return;
			if (!apiKey || typeof apiKey != 'string' || apiKey.length < 1) return;
			const slash = serverURL.endsWith('/') ? '' : '/';
			const endpoint = `${serverURL}${slash}auth/me`;
			fetch(endpoint, {
				method: 'GET',
				headers: {
					'Content-Type': 'application/json',
					'Authorization': `Bearer ${apiKey}`
				}
			}).then(response => {
				if (!response.ok) {
					throw new Error(`Server responded with status ${response.status}`);
				}
				return response.json();
			}).then(data => {
				if(!data || typeof data != 'object'){
					throw new Error('Invalid response data');
				}
				if(!data.ok || !data.authenticated){
					throw new Error('Authentication failed');
				}
				if(!data.public_key || typeof data.public_key != 'string'){
					throw new Error('No public key in response');
				}
				this.publicKey = data.public_key;
				this.userFiat = data?.user_fiat || null;
				if(this.userFiat && typeof this.userFiat == 'string'){
					$('#user_preferred_fiat').val(this.userFiat);
				}else{
					$('#user_preferred_fiat').val('');
				}
			}).catch(error => {
				console.error('Error fetching user public key using api key. ', error);
			});
		}catch(e){
			console.error('Error fetching user public key using api key. ', e);
		}
	}

	loadState() {
		chrome.storage.local.get(['settings', 'thread_sort_mode'], (result) => {
			if (chrome.runtime.lastError) {
				console.error('Error loading state:', chrome.runtime.lastError);
				return;
			}
			this.state.settings = result.settings || {};
			this.state.thread_sort_mode = result.thread_sort_mode || 'date_desc'; // default to newest sort mode
			// Ensure all default settings are present if missing
			for (let key in this.settingsDefault) {
				if (!(key in this.state.settings)) {
					this.state.settings[key] = this.settingsDefault[key];
				}
			}
			this.authenticate();
		});
	}

	saveState() {
		chrome.storage.local.set({
			settings: this.state.settings || {},
			thread_sort_mode: this.state.thread_sort_mode || 'date_desc',
		}, () => {
			if (chrome.runtime.lastError) {
				console.error('Error saving state:', chrome.runtime.lastError);
			}
		});
	}

	applyFontSizeSetting() {
		const font_size = this.getSetting('font_size');
		if (!font_size || typeof font_size != 'string' || font_size.length < 1) return;
		// validate that the font size ends with em and is a number from 0.5 to 1.5
		const font_size_num = parseFloat(font_size.replace('em', ''));
		if (isNaN(font_size_num) || font_size_num < 0.5 || font_size_num > 1.5 || !font_size.endsWith('em')) return;
		$('body').css({ fontSize: font_size });
	}

	getSetting(key) {
		if ('settings' in this.state && this.state.settings && typeof this.state.settings == 'object' && key in this.state.settings) {
			return this.state.settings[key];
		}
		return this.settingsDefault[key] || null;
	}

	updateSettings(newSettings) {
		let validSettings = {};
		let invalidParams = [];
		for (let key in newSettings) {
			if (this.settingsSchema[key]) {
				if (typeof newSettings[key] === this.settingsSchema[key]) {
					validSettings[key] = newSettings[key];
				} else if (this.settingsSchema[key] === 'number' && !isNaN(newSettings[key] * 1)) {
					validSettings[key] = newSettings[key] * 1;
				} else if (this.settingsSchema[key] === 'boolean' && ["true", "false"].indexOf(newSettings[key].toString().toLowerCase()) > -1) {
					validSettings[key] = newSettings[key].toString().toLowerCase() === 'true' ? true : false;
				} else {
					invalidParams.push(key);
				}
			} else {
				invalidParams.push(key);
			}
		}

		if (invalidParams.length > 0) {
			const invStr = invalidParams.join(', ');
			this.feed(`Invalid setting or type for parameter(s): ${invStr}`, true);
		} else {
			// merge partial settings with existing settings
			this.state.settings = { ...this.state.settings, ...validSettings };
			this.saveState();
			this.feed("Settings updated.")
		}
	}

	buildSettingsForm() {
		$('#reply-container').empty().append('<h2>Extension Settings</h2>');

		// Create cancel button
		const cancelIcon = bsi('x-lg') || '❌';
		const cancelLink = $(`<a href="#" class="btn btn-sm btn-secondary float-end" id="exit_settings">${cancelIcon} Cancel</a>`);
		cancelLink.on('click', e => {
			e.preventDefault();
			this.getThreads();
		});
		$('#reply-container').append(cancelLink, '<br><br>');

		const settingsForm = $(`<form></form>`);
		for (var key in this.settingsDefault) {
			var input = null;
			var label = `<label for="${key}" title="${(this.settingsDescriptions[key] || '')}">${key.replace(/_/g, ' ').toUpperCase()}</label>`;
			if (key == 'font_size') { // font size dropdown
				input = $(`<select name="font_size"></select>`);
				const options = ['0.5em', '0.6em', '0.7em', '0.8em', '0.9em', '1em', '1.1em', '1.2em', '1.3em', '1.4em', '1.5em'];
				const dflt = this.state.settings?.[key] || this.settingsDefault?.[key] || null;
				for (var j = 0; j < options.length; j++) {
					const opt = options[j];
					const selected = dflt == opt ? ' selected' : '';
					input.append(`<option value="${opt}"${selected}>${opt}</option>`);
				}
			} else {
				const typ = typeof this.settingsDefault[key] === 'number' ? 'number' : 'text';
				const val = this.state.settings?.[key] || this.settingsDefault[key];
				input = $(`<input type="${typ}" title="${(this.settingsDescriptions[key] || '')}" name="${key}" value="${val}">`);
			}
			if (!input) continue; // skip if no input
			if (label) settingsForm.append(label, '<br>');
			input.addClass('form-control');
			settingsForm.append(input, '<br>');
		}

		// append submit button
		settingsForm.append(`<br><br><input type="submit" value="Save Settings" class="btn-primary form-control"><br><br>`);
		settingsForm.on('submit', (e) => {
			e.preventDefault();
			for (var key in this.settingsDefault) {
				const input = settingsForm.find(`[name="${key}"]`);
				if (input.length < 1) continue;
				const val = input.val();
				if (Array.isArray(this.settingsDefault[key])) {
					const checkedBoxes = input.find('input[type="checkbox"]:checked');
					const checkedValues = [];
					checkedBoxes.each((i, el) => {
						checkedValues.push($(el).val());
					});
					this.updateSettings({ [key]: checkedValues });
				} else if (this.settingsSchema?.[key] == 'boolean') {
					this.updateSettings({ [key]: input.is(':checked') });
				} else {
					this.updateSettings({ [key]: val });
				}
			}
			this.saveState();
			this.applyFontSizeSetting();
			$('#exit_settings').trigger('click');
		});
		$('#reply-container').append(settingsForm);

		// Display extension version info
		if (this.version && typeof this.version == 'string') {
			$('#reply-container').append(`<div class="text-muted small" style="margin-top:10px;">Extension Version: ${this.version}</div>`);
		}else{
			$('#reply-container').append(`<div class="text-muted small" style="margin-top:10px;">Extension Version: Unknown</div>`);
		}
	}
}
